import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";
import { SystemAnalysisData } from "@/types/analysis";

interface PerformanceTabProps {
  systemData: SystemAnalysisData;
}

export default function PerformanceTab({ systemData }: PerformanceTabProps) {
  const { t } = useLanguage();
  return (
    <>
      {/* Technician Performance */}
      <div className='bg-white rounded-lg shadow mb-8'>
        <div className='px-6 py-4 border-b border-gray-200'>
          <h2 className='text-lg font-semibold'>
            {t("analysis.technicianPerformance")}
          </h2>
        </div>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("admin.technicians")}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.totalTickets")}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.avgResolutionTime")}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.completionRate")}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {systemData.technicianStats.map((tech) => (
                <tr key={tech.technicianId}>
                  <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                    {tech.technicianName}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {tech.totalTickets}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {tech.avgResolutionTime.toFixed(1)}h
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    <span
                      className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                        tech.completionRate >= 80
                          ? "bg-green-100 text-green-800"
                          : tech.completionRate >= 60
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-red-100 text-red-800"
                      }`}
                    >
                      {tech.completionRate.toFixed(1)}%
                    </span>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Severity Analysis */}
      <div className='bg-white rounded-lg shadow mb-8'>
        <div className='px-6 py-4 border-b border-gray-200'>
          <h2 className='text-lg font-semibold'>
            {t("analysis.severityAnalysis")}
          </h2>
        </div>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.severityLevel")}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.ticketCount")}
                </th>
                <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.avgResolutionTime")}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {systemData.severityStats.map((severity) => (
                <tr key={severity.severity}>
                  <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                    <span
                      className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                        severity.severity >= 4
                          ? "bg-red-100 text-red-800"
                          : severity.severity >= 3
                          ? "bg-orange-100 text-orange-800"
                          : severity.severity >= 2
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-green-100 text-green-800"
                      }`}
                    >
                      Level {severity.severity}
                    </span>
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {severity.count}
                  </td>
                  <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                    {severity.avgResolutionTime.toFixed(1)}h
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </>
  );
}
